/* FILE: mpick.c                                (D. Tottingham  03/23/91)

This is a collection of C helper functions that manage a real-time picker for
xdetect.  All functions have been written and compiled medium model.  The
following functions are included:

pk_get_first_arrival ()         get first p-arrival
pk_get_first_magnitude ()       get first magnitude
pk_get_last_arrival ()          get last p-arrival
pk_get_last_magnitude ()        get last magnitude
pk_get_next_arrival ()          get next p-arrival
pk_get_next_magnitude ()        get next magnitude
pk_get_narrivals ()             get number of arrivals
pk_get_nmagnitudes ()           get number of magnitudes
pk_initialize ()                initialize pick module
pk_pick_arrivals ()             pick arrivals using trigger information
pk_pick_magnitudes ()           pick magnitudes using coda duration
pk_reset_picks ()               reset pick queue

EXTERNAL FUNCTIONS CALLED:

er_abort ()                     display an error message then quit
q_dequeue ()                    dequeue a data link from a data queue
q_enqueue ()                    enqueue a data link on a data queue
q_initialize ()                 initialize a data queue
q_insert_link ()                insert a data link into a data queue
suds_initialize ()              initialize a suds structure
suds_initialize_tag ()          initialize a suds structtag
st_get_authority ()             get authority
st_get_station ()               get station from station queue
t_get_head_trigger ()           get trigger information at head of channel queue
t_get_next_trigger ()           get next trigger from channel queue
u_timestamp ()                  get timestamp

HISTORY:
   none

*/



/*************************************************************************
                            INCLUDE FILES

*************************************************************************/
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>

#include "mconst.h"
#include "mdemux.h"
#include "merror.h"
#include "mqueue.h"
#include "msudsini.h"
#include "mstation.h"
#include "mtrigger.h"
#include "mutils.h"
#include "xdetect.h"


/*************************************************************************
                              GLOBAL DATA

**************************************************************************/
PRIVATE Q_QUEUE arrival_queue, magnitude_queue;
PRIVATE Q_LINK * arrival_ptr, * magnitude_ptr;
PRIVATE unsigned int n_arrivals, n_magnitudes;


/*=======================================================================*
 *                           insert_arrival                              *
 *=======================================================================*/
/* Insert p-arrival in arrival queue based on arrival time.              */

PRIVATE
void insert_arrival (arrival)
Q_ARRIVAL far * arrival;
{
   Q_LINK *current, *previous;
   Q_TYPE type;
   FLAG done;

   type.arrival = arrival;
   previous = NULL;
   for (current = arrival_queue.head, done = FALSE; !done && current != NULL;
        current = current->next) {
      if (arrival->info.time < current->type.arrival->info.time) {
         q_insert_link (&arrival_queue, previous, type);
         done = TRUE;
      } else previous = current;
   }
   if (!done) q_enqueue (&arrival_queue, type);
}

/*=======================================================================*
 *                            process_coda                               *
 *=======================================================================*/
/* Compute signal value for given channel (specified in the arrival) and
   check against noise.  If signal/noise < 2, compute coda duration and
   return it to the caller.                                              */

PRIVATE
double process_coda (arrival_ptr, buffer_ptr)
Q_LINK *arrival_ptr;
Q_BUFFER far * buffer_ptr;
{
   Q_ARRIVAL far * arrival;
   long channel_offset, start_offset, i, max_index;
   double deltat;
   int ns;

   arrival = arrival_ptr->type.arrival;

   start_offset = 0;
   if ((deltat = arrival->info.time - buffer_ptr->info.begintime) > 0.0)
      start_offset = (long) (deltat * buffer_ptr->info.dig_rate);

   max_index = 0;
   arrival->signal = 0;
   channel_offset = buffer_ptr->info.blocksize * arrival->channel;
   for (i = (channel_offset + start_offset); i < (buffer_ptr->info.blocksize + channel_offset); i++) {
      ns = abs (buffer_ptr->data[i] - buffer_ptr->info.dc_offset);
      if (ns > arrival->signal) {
         arrival->signal = ns;
         max_index = i;
      }
      if ((arrival->signal / arrival->noise) > 2)
         return (0.0);
   }
   deltat = ((double)(max_index - channel_offset)) / buffer_ptr->info.dig_rate;
/* if (Debug_enabled) printf ("deltat = %lf\n", deltat);  */
   return ( deltat + buffer_ptr->info.begintime - arrival->info.time );
}

/*=======================================================================*
 *                        pk_get_first_arrival                           *
 *=======================================================================*/
/* Get first p-arrival from arrival queue.                               */

PUBLIC
Q_ARRIVAL far * pk_get_first_arrival ()
{
   arrival_ptr = arrival_queue.head;
   if (arrival_ptr != NULL)
      return (arrival_ptr->type.arrival);
   else return (NULL);
}

/*=======================================================================*
 *                       pk_get_first_magnitude                          *
 *=======================================================================*/
/* Get first magnitude from magnitude queue.                             */

PUBLIC
Q_MAGNITUDE far * pk_get_first_magnitude ()
{
   magnitude_ptr = magnitude_queue.head;
   if (magnitude_ptr != NULL)
      return (magnitude_ptr->type.magnitude);
   else return (NULL);
}

/*=======================================================================*
 *                         pk_get_last_arrival                           *
 *=======================================================================*/
/* Get last p-arrival from arrival queue.                                */

PUBLIC
Q_ARRIVAL far * pk_get_last_arrival ()
{
   if (arrival_queue.tail != NULL)
      return (arrival_queue.tail->type.arrival);
   else return (NULL);
}

/*=======================================================================*
 *                        pk_get_last_magnitude                          *
 *=======================================================================*/
/* Get last magnitude from magnitude queue.                              */

PUBLIC
Q_MAGNITUDE far * pk_get_last_magnitude ()
{
   if (magnitude_queue.tail != NULL)
      return (magnitude_queue.tail->type.magnitude);
   else return (NULL);
}

/*=======================================================================*
 *                         pk_get_next_arrival                           *
 *=======================================================================*/
/* Get next p-arrival from arrival queue.                                */

PUBLIC
Q_ARRIVAL far * pk_get_next_arrival ()
{
   arrival_ptr = arrival_ptr->next;
   if (arrival_ptr != NULL)
      return (arrival_ptr->type.arrival);
   else return (NULL);
}

/*=======================================================================*
 *                        pk_get_next_magnitude                          *
 *=======================================================================*/
/* Get next magnitude from magnitude queue.                              */

PUBLIC
Q_MAGNITUDE far * pk_get_next_magnitude ()
{
   magnitude_ptr = magnitude_ptr->next;
   if (magnitude_ptr != NULL)
      return (magnitude_ptr->type.magnitude);
   else return (NULL);
}

/*=======================================================================*
 *                          pk_get_narrivals                             *
 *=======================================================================*/
/* Get number of p-arrivals in arrival queue.                            */

PUBLIC
unsigned int pk_get_narrivals ()
{
   return (n_arrivals);
}

/*=======================================================================*
 *                         pk_get_nmagnitudes                            *
 *=======================================================================*/
/* Get number of magnitudes in magnitude queue.                          */

PUBLIC
unsigned int pk_get_nmagnitudes ()
{
   return (n_magnitudes);
}

/*=======================================================================*
 *                            pk_initialize                              *
 *=======================================================================*/
/* Initialize the pick module.                                           */

PUBLIC
void pk_initialize ()
{
   q_initialize (&arrival_queue);
   q_initialize (&magnitude_queue);
   n_arrivals = n_magnitudes = 0;
}

/*=======================================================================*
 *                          pk_pick_arrivals                             *
 *=======================================================================*/
/* Pick arrivals using trigger information.                              */

PUBLIC
void pk_pick_arrivals ()
{
   Q_CHANNEL far *trig_ptr;
   Q_LINK *link_ptr;
   Q_ARRIVAL far *new_arrival;
   Q_TYPE type;
   double first_arrival;

   /* Get new picks */
   for (trig_ptr = t_get_head_trigger(); trig_ptr != NULL; trig_ptr = t_get_next_trigger()) {
      if ((st_get_station(trig_ptr->channel))->in_masterlist &&
          trig_ptr->info.trig_value) {

         /* Initialize a new arrival structure */
         new_arrival = (Q_ARRIVAL far *) _fmalloc (sizeof(Q_ARRIVAL));
         if (new_arrival == NULL) er_abort (PK_NO_STORAGE);
         suds_initialize_tag (FEATURE, &new_arrival->structtag);
         suds_initialize (FEATURE, &new_arrival->info);

         /* Fill it in */
         new_arrival->channel = trig_ptr->channel;
         new_arrival->signal = trig_ptr->signal;
         new_arrival->noise = trig_ptr->noise;
         new_arrival->info.obs_phase = 50;      /* p first arrival */
         new_arrival->info.fe_name = trig_ptr->info.tr_name;
         new_arrival->info.data_source = 'r';
         new_arrival->info.tim_qual = '4';
         new_arrival->info.time = trig_ptr->info.trig_time;
         new_arrival->info.pick_authority = st_get_authority ();
         new_arrival->info.time_of_pick = u_timestamp ();

         insert_arrival (new_arrival);
         n_arrivals++;
      }
   }

   /* Compute relative arrivals based on first arrival */
   first_arrival = arrival_queue.head->type.arrival->info.time;
   for (link_ptr = arrival_queue.head; link_ptr != NULL; link_ptr = link_ptr->next)
      link_ptr->type.arrival->rel_arrival = link_ptr->type.arrival->info.time - first_arrival;
}

/*=======================================================================*
 *                         pk_pick_magnitudes                            *
 *=======================================================================*/
/* Pick magnitudes using coda duration.                                  */

PUBLIC
void pk_pick_magnitudes ()
{
   Q_BUFFER far *b_head;
   Q_LINK *a_head, *m_head;
   Q_MAGNITUDE far *new_magnitude;
   Q_TYPE type;
   double coda_duration;
   FLAG found;

   /* Check the arrival queue */
   if (arrival_queue.head == NULL) return;

   for (b_head = dm_get_first_buffer(); b_head != NULL; b_head = dm_get_next_buffer())
      for (a_head = arrival_queue.head; a_head != NULL; a_head = a_head->next) {
         for (found = FALSE, m_head = magnitude_queue.head;
              m_head != NULL && !found; m_head = m_head->next)
            if (m_head->type.magnitude->channel == a_head->type.arrival->channel)
               found = TRUE;

         if (! found && (coda_duration = process_coda (a_head, b_head)) > 0.0) {
/*
if (Debug_enabled)
   printf ("ch, coda_duration = %3d, %lf\n", a_head->type.arrival->channel, coda_duration);
*/
            /* Initialize a new magnitude structure */
            new_magnitude = (Q_MAGNITUDE far *) _fmalloc (sizeof(Q_MAGNITUDE));
            if (new_magnitude == NULL) er_abort (PK_NO_STORAGE);
            suds_initialize_tag (FEATURE, &new_magnitude->structtag);
            suds_initialize (FEATURE, &new_magnitude->info);

            /* Fill it in */
            new_magnitude->channel = a_head->type.arrival->channel;
            new_magnitude->info.obs_phase = 2;      /* f finis */
            new_magnitude->info.fe_name = a_head->type.arrival->info.fe_name;
            new_magnitude->info.data_source = 'r';
            new_magnitude->info.tim_qual = '4';
            new_magnitude->info.time = coda_duration;
            new_magnitude->info.pick_authority = a_head->type.arrival->info.pick_authority;
            new_magnitude->info.time_of_pick = u_timestamp ();

            type.magnitude = new_magnitude;
            q_enqueue (&magnitude_queue, type);
            n_magnitudes++;
         }
      }
}

/*=======================================================================*
 *                            pk_reset_picks                             *
 *=======================================================================*/
/* Reset pick queues.                                                    */

PUBLIC
void pk_reset_picks ()
{
   Q_TYPE type;

   while (q_dequeue (&arrival_queue, &type))
      _ffree (type.arrival);
   n_arrivals = 0;

   while (q_dequeue (&magnitude_queue, &type))
      _ffree (type.magnitude);
   n_magnitudes = 0;
}
